package de.communardo.confluence.plugins.metadataintegration.blueprint;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;

import de.communardo.confluence.plugins.metadataintegration.type.SmileyDataObject;
import de.communardo.confluence.plugins.metadataintegration.type.SmileyMetadataFieldType;
import de.communardo.confluence.plugins.metadataintegration.type.SmileyMetadataFieldType.SmileyStyle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintHomePageCreateEvent;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.communardo.confluence.metadata.DataObject;
import com.communardo.confluence.metadata.MetadataException;
import com.communardo.confluence.metadata.MetadataField;
import com.communardo.confluence.metadata.MetadataFieldType;
import com.communardo.confluence.metadata.MetadataImportException;
import com.communardo.confluence.metadata.MetadataSet;
import com.communardo.confluence.metadata.fieldtype.Transferable;
import com.communardo.confluence.metadata.service.MetadataFieldTypeManager;
import com.communardo.confluence.metadata.service.MetadataManager;
import com.google.gson.Gson;

/**
 * Event listener which adds a Metadata set to a space that is create by our blueprint. It also
 * assigns the set to the home page of this space.
 *
 * @author Communardo Products GmbH
 */
@Named
public class MetadataIntegrationBlueprintEventListener implements DisposableBean {

    private static final String FIELD_TYPE_TEXT = "TEXT";

    private static final String FIELD_TYPE_RADIOBUTTON = "RADIOBUTTON";

    private static final String OPTION_FAIL = "FAIL";

    private static final String OPTION_SUCCESS = "SUCCESS";

    private static final Logger LOGGER = LoggerFactory.getLogger(MetadataIntegrationBlueprintEventListener.class);

    private static final String BLUEPRINT_MODULE_KEY = "de.communardo.confluence.plugins.metadata-integration:metadata-integration-space-blueprint";

    @ComponentImport
    private final ContentPropertyManager contentPropertyManager;
    @ComponentImport
    private final SpaceManager spaceManager;
    @ComponentImport
    private final EventPublisher eventPublisher;
    @ComponentImport
    private final MetadataFieldTypeManager metadataFieldTypeManager;
    @ComponentImport
    private final MetadataManager metadataManager;

    @Inject
    public MetadataIntegrationBlueprintEventListener(SpaceManager spaceManager,
                                                     ContentPropertyManager contentPropertyManager, EventPublisher eventPublisher,
                                                     MetadataFieldTypeManager metadataFieldTypeManager, MetadataManager metadataManager) {
        this.contentPropertyManager = contentPropertyManager;
        this.spaceManager = spaceManager;
        this.eventPublisher = eventPublisher;
        this.metadataFieldTypeManager = metadataFieldTypeManager;
        this.metadataManager = metadataManager;

        this.eventPublisher.register(this);
    }

    @Override
    public void destroy() throws Exception {
        // unregister the listener if the app is uninstalled or disabled.
        this.eventPublisher.unregister(this);
    }

    @EventListener
    public void onSpaceHomePageCreated(SpaceBlueprintHomePageCreateEvent homePageCreateEvent) {
        // if the space has been created with our blueprint we add our Metadata set to the space
        // and its home page
        if (BLUEPRINT_MODULE_KEY.equals(homePageCreateEvent.getSpaceBlueprint().getModuleCompleteKey())) {
            Space space = homePageCreateEvent.getSpace();
            try {
                // add Metadata structure to space
                MetadataSet metadataSet = addMetadataStructureToSpace(space);
                // add the Metadata set to the space home page
                metadataManager.saveContentMetadataSet(space.getHomePage(), metadataSet);
                // add a value for each field of the set to the home page
                for (MetadataField field : metadataSet.getMetadataFields()) {
                    addMetadataValueToHomeSpacePage(space.getHomePage(), field);
                }
            } catch (MetadataImportException e) {
                LOGGER.error("Adding Metadata structure to space '{}' failed", space.getKey(), e);
            } catch (MetadataException e) {
                LOGGER.error("Adding Metadata set to the home page of space created by blueprint failed", e);
            }

        }
    }

    private void addMetadataValueToHomeSpacePage(Page homePage, MetadataField field) {
        MetadataFieldType fieldType = field.getType();
        // we are using the JSON export format to provide the value as Metadata devs won't change it
        // to not break things. The export format is a feature provided via the Transferable
        // interface
        if (!(fieldType instanceof Transferable)) {
            LOGGER.error("Unsupported Metadata field '{}' of type '{}'", field.getKey(), fieldType);
            return;
        }
        Transferable transferableFieldType = (Transferable) fieldType;
        DataObject dataObject = null;
        try {
            if (FIELD_TYPE_TEXT.equals(fieldType.getName())) {
                // export format is just the text value
                dataObject = transferableFieldType.createDataObject("Space home page successfully created.");
            } else if (FIELD_TYPE_RADIOBUTTON.equals(fieldType.getName())) {
                // export format of this type is a JSON serialized array which contains the name of
                // the selected option
                String exportFormat = new Gson().toJson(Arrays.asList(OPTION_SUCCESS));
                dataObject = transferableFieldType.createDataObject(exportFormat);
            } else if (SmileyMetadataFieldType.FIELD_TYPE_NAME.equals(fieldType.getName())) {
                dataObject = SmileyDataObject.getLaughingSmiley(SmileyStyle.YELLOW) ;
            } else {
                LOGGER.error("Unsupported Metadata field '{}' of type '{}'", field.getKey(), fieldType);
            }
            if (dataObject != null) {
                metadataManager.saveContentMetadataValue(homePage, field, dataObject);
            }
        } catch (MetadataException | MetadataImportException e) {
            LOGGER.error("Adding Metadata value of field '{}' to space home page failed", field.getKey(), e);
        }
    }

    private MetadataSet addMetadataStructureToSpace(Space space) throws MetadataImportException {
        // create the Metadata structure by first creating the fields
        List<MetadataField> metadataFields = new ArrayList<>();
        metadataFields.add(createStatusMetadataField(space));
        metadataFields.add(createStatusMessageMetadataField(space));
        metadataFields.add(createSmileyMetadataField(space));

        // create a set
        MetadataSet exampleSet = new MetadataSet();
        exampleSet.setDefaultSet(true);
        exampleSet.setKey("metadataintegrationset");
        exampleSet.setTitle("Metadata integration set");
        exampleSet.setSpace(space);
        // add the fields to the set
        exampleSet.setMetadatas(metadataFields);

        // store it
        metadataManager.saveMetadataSet(exampleSet);
        return exampleSet;
    }

    private MetadataField createSmileyMetadataField(Space space) throws MetadataImportException {
        MetadataField metadataField = createMetadataField("Smiley mood", "metadataintegrationsmileymood", getFieldType(SmileyMetadataFieldType.FIELD_TYPE_NAME), space, null);
        ((SmileyMetadataFieldType) metadataField.getType()).setYellowTypeConfiguration();
        return metadataField;
    }

    private MetadataField createStatusMetadataField(Space space) throws MetadataImportException {
        // the status is a single select (RADIOBUTTON) Metadata field with 2 options
        // prepare the field configuration which holds the options and convert it to the JSON
        // export format of this Metadata field type (see createMetadataField for background)
        List<String> options = Arrays.asList(OPTION_SUCCESS, OPTION_FAIL);
        String fieldConfigExportFormat = new Gson().toJson(options);
        // note: the field key will be used in the CQL field for for the Metadata field. Therefore,
        // certain characters are not allowed. Safest is to just use ASCII letters.
        return createMetadataField("Status", "metadataintegrationstatus", getFieldType(FIELD_TYPE_RADIOBUTTON), space,
                fieldConfigExportFormat);
    }

    private MetadataField createStatusMessageMetadataField(Space space) throws MetadataImportException {
        // the status message is just a simple text field which has no configuration
        return createMetadataField("Status message", "metadataintegrationstatusmessage", getFieldType(FIELD_TYPE_TEXT),
                space, null);
    }

    private MetadataFieldType getFieldType(String fieldTypeName) throws MetadataImportException {
        MetadataFieldType fieldType = metadataFieldTypeManager.getFieldTypeByName(fieldTypeName);
        if (fieldType == null) {
            throw new MetadataImportException("Field type with name " + fieldTypeName + " doesn't exist");
        }
        return fieldType;
    }

    private MetadataField createMetadataField(String fieldTitle, String fieldKey, MetadataFieldType fieldType,
                                              Space space, String fieldConfigInExportFormat) throws MetadataImportException {
        MetadataField metadataField = new MetadataField();
        metadataField.setKey(fieldKey);
        metadataField.setTitle(fieldTitle);
        metadataField.setType(fieldType);
        metadataField.setSpace(space);
        metadataField = metadataManager.saveMetadataField(metadataField);
        // save configuration of the field if needed. Using the JSON export format as Metadata devs
        // won't change it to not break things.
        if (fieldConfigInExportFormat != null) {
            // take the field type instance which is linked with the newly created field
            fieldType = metadataField.getType();
            if (fieldType instanceof Transferable) {
                ((Transferable) fieldType).importTypeConfiguration(fieldConfigInExportFormat);
            }
        }
        return metadataField;
    }

}
