package de.communardo.confluence.plugins.metadataintegration.type;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.setup.bandana.ConfluenceBandanaContext;
import com.atlassian.confluence.util.HtmlUtil;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.querylang.fields.FieldHandler;
import com.atlassian.querylang.fields.UISupport;
import com.atlassian.querylang.fields.ValueType;
import com.atlassian.sal.api.message.I18nResolver;
import com.communardo.confluence.metadata.DataObject;
import com.communardo.confluence.metadata.MetadataExportException;
import com.communardo.confluence.metadata.MetadataField;
import com.communardo.confluence.metadata.MetadataFieldType;
import com.communardo.confluence.metadata.MetadataImportException;
import com.communardo.confluence.metadata.MetadataSet;
import com.communardo.confluence.metadata.cql.MetadataSchemaField;
import com.communardo.confluence.metadata.fieldtype.DefaultValueSupport;
import com.communardo.confluence.metadata.fieldtype.Transferable;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import de.communardo.confluence.plugins.metadataintegration.type.cql.SmileyMetadataFieldHandler;
import org.apache.commons.lang.StringUtils;

import javax.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;


/**
 * A Metadata field type which allows selecting a smiley as Metadata value :)
 *
 * @author Communardo Products GmbH
 */
public class SmileyMetadataFieldType extends MetadataFieldType implements Transferable, DefaultValueSupport {

    public enum SmileyStyle {
        BLUE,
        YELLOW;

        @Override
        public String toString() {
            return this.name().toLowerCase(Locale.ENGLISH);
        }
    }

    private static final String FIELD_CONFIG_SMILEY_STYLE = "smileytype";
    public static final String FIELD_TYPE_NAME = "SMILEY";


    private final I18nResolver i18nResolver;
    private final BandanaManager bandanaManager;
    private final ContentPropertyManager contentPropertyManager;

    @Inject
    public SmileyMetadataFieldType(@ComponentImport I18nResolver i18nResolver,
                                   @ComponentImport BandanaManager bandanaManager,
                                   @ComponentImport ContentPropertyManager contentPropertyManager) {
        this.bandanaManager = bandanaManager;
        this.i18nResolver = i18nResolver;
        this.contentPropertyManager = contentPropertyManager;
    }

    /**
     * @return a unique name for the new type
     */
    @Override
    public String getName() {
        return FIELD_TYPE_NAME;
    }

    /**
     * Create the HTML which should be used when a smiley Metadata value should be rendered.
     *
     * @param dataObject          the smiley Metadata value data object
     * @param contentEntityObject the content to which the Metadata value is assigned
     * @return the HTML representing the render view of the Metadata value
     */
    @Override
    public String renderData(DataObject dataObject, ContentEntityObject contentEntityObject) {
        Map<String, Object> defaultVelocityContext = MacroUtils.defaultVelocityContext();
        defaultVelocityContext.put("smileyStyle", loadSmileyStyle().toString());
        defaultVelocityContext.put("currentSmiley", dataObject);

        return VelocityUtils.getRenderedTemplate("template/smiley_type_view.vm", defaultVelocityContext);
    }

    /**
     * Create the HTML which should be used when editing the value of a smiley Metadata field.
     *
     * @param contentEntityObject the page to which the Metadata field should be assigned or is already assigned
     * @return the HTML representing the edit view of the Metadata value
     */
    @Override
    public String renderEdit(ContentEntityObject contentEntityObject) {
        return renderEdit(loadDataObject(contentEntityObject));
    }

    /**
     * Create the HTML which should be used when editing the value of a smiley Metadata field.
     *
     * @param dataObject object representing the Metadata value to be rendered
     * @return the HTML representing the edit view of the Metadata value
     */
    @Override
    public String renderEdit(DataObject dataObject) {
        return renderSmileyValueEditView(getMetadataField(), dataObject);
    }

    @Override
    public FieldHandler getCqlSearchFieldHandler(MetadataSchemaField schemaField) {
        return new SmileyMetadataFieldHandler(schemaField, true);
    }

    /**
     * Create the HTML which should be injected into the form for creating a new or editing an
     * existing Metadata field of this type. This allows making the field configurable by adding
     * additional input elements. The configuration will be available in the paramter map passed to
     * method {@link #onSaveMetadataField(Map)}.
     *
     * @return optional HTML to include when admins or space admins create or edit a Metadata field
     * of this type
     */
    @Override
    public String renderAdmin() {
        Map<String, Object> defaultVelocityContext = MacroUtils.defaultVelocityContext();
        defaultVelocityContext.put("currentSmileyStyle", loadSmileyStyle().toString());
        defaultVelocityContext.put("BLUE_STYLE", SmileyStyle.BLUE.toString());
        defaultVelocityContext.put("YELLOW_STYLE", SmileyStyle.YELLOW.toString());

        return VelocityUtils.getRenderedTemplate("template/smiley_type_admin.vm", defaultVelocityContext);
    }

    /**
     * Save the configuration of the Metadata field.
     *
     * @param parameters the request parameters passed when the Metadata field linked to this type was
     *                   saved
     * @return whether the configuration was saved successfully
     */
    @Override
    public boolean onSaveMetadataField(Map parameters) {
        Map<String, String> typeConfig = new HashMap<>();

        String[] style = (String[]) parameters.get("smileyStyle");
        if (style != null && style.length > 0) {
            typeConfig.put(FIELD_CONFIG_SMILEY_STYLE, resolveSmileyStyle(style[0].trim()).toString());
        }

        updateTypeConfiguration(typeConfig);
        return true;
    }

    @Override
    public void onRemoveMetadataField() {
        bandanaManager.removeValue(new ConfluenceBandanaContext(), getTypeConfigBandanaStorarageKey());
    }

    private String getTypeConfigBandanaStorarageKey() {
        return getTypeConfigBandanaStorarageKey(getMetadataField());
    }

    private String getTypeConfigBandanaStorarageKey(MetadataField metadataField) {
        if (metadataField == null) {
            return null;
        }
        return "SMILEYTYPECONFIG-" + (metadataField.getSpace() != null ? metadataField.getSpace().getKey() : "global")
                + metadataField.getKey();
    }

    private void updateTypeConfiguration(Map<String, String> smileyConfig) {
        String bandanaStorageKey = getTypeConfigBandanaStorarageKey();
        if (bandanaStorageKey != null) {
            Gson gson = new Gson();
            bandanaManager.setValue(new ConfluenceBandanaContext(), bandanaStorageKey, gson.toJson(smileyConfig));
        }
    }

    private Map<String, String> loadTypeConfiguration() {
        return loadTypeConfiguration(getMetadataField());
    }

    private Map<String, String> loadTypeConfiguration(MetadataField metadataField) {
        String storedSmileyConfig = null;
        String bandanaStorageKey = getTypeConfigBandanaStorarageKey(metadataField);
        if (bandanaStorageKey != null) {
            storedSmileyConfig = (String) bandanaManager.getValue(new ConfluenceBandanaContext(), bandanaStorageKey);
        }
        TypeToken<Map<String, String>> typeToken = new TypeToken<Map<String, String>>() {
        };
        if (StringUtils.isBlank(storedSmileyConfig)) {
            return new HashMap<>();
        }
        return new Gson().fromJson(storedSmileyConfig, typeToken.getType());
    }

    public SmileyStyle loadSmileyStyle() {
        return loadSmileyStyle(getMetadataField());
    }

    private SmileyStyle loadSmileyStyle(MetadataField metadataField) {
        return resolveSmileyStyle(loadTypeConfiguration(metadataField).get(FIELD_CONFIG_SMILEY_STYLE));
    }

    private SmileyStyle resolveSmileyStyle(String style) {
        if (SmileyStyle.YELLOW.toString().equals(style)) {
            return SmileyStyle.YELLOW;
        }
        // blue is default
        return SmileyStyle.BLUE;
    }

    @Override
    public Collection<String> getSearchIndexValue(ContentEntityObject contentEntityObject) {
        DataObject dataObject = loadDataObject(contentEntityObject);
        return getSearchIndexValue(dataObject);
    }

    @Override
    public Collection<String> getSearchIndexValue(DataObject dataObject) {

        List<String> result = new ArrayList<>();
        if (dataObject instanceof SmileyDataObject) {
            result.add(String.valueOf(((SmileyDataObject) dataObject).getId()));
        }
        return result;
    }

    /**
     * This method will be called for a Metadata field of this type when a user publishes a page /
     * blogpost. The passed map contains the form data the user provided via the form elements
     * contained in the HTML returned by {@link #renderEdit(DataObject)}. The returned DataObject
     * will be passed to {@link #saveDataObject(ContentEntityObject, DataObject)}.
     *
     * @param map the map with the form data provided by the user
     * @return an object representing the value of this Metadata field
     */
    @Override
    public DataObject createDataObject(Map map) {
        String[] object = (String[]) map.get("metadatavalue-" + getMetadataField().getId());
        if (object != null && object.length > 0) {
            return SmileyDataObject.fromStorageFormat(object[0]);
        }
        return null;
    }

    /**
     * Load an object representing the Metadata value of this field type that is assigned to the
     * given content.
     *
     * @param contentEntityObject the content for which the value should be loaded
     * @return the object representing the loaded Metadata value, can be null if no value is
     * assigned to the content
     */
    @Override
    public DataObject loadDataObject(ContentEntityObject contentEntityObject) {
        if (contentEntityObject != null) {

            String loadContentMetadataValue = contentPropertyManager.getStringProperty(contentEntityObject,
                    getMetadataField().getKey());
            return SmileyDataObject.fromStorageFormat(loadContentMetadataValue);
        }
        return null;
    }

    /**
     * Assign a value for the Metadata field which is linked to this type to the given content.
     *
     * @param contentEntityObject the content to which the value should be assigned
     * @param dataObject          object representing the Metadata value to assign
     */
    @Override
    public void saveDataObject(ContentEntityObject contentEntityObject, DataObject dataObject) {
        if (dataObject instanceof SmileyDataObject) {
            contentPropertyManager.setStringProperty(contentEntityObject, getMetadataField().getKey(),
                    ((SmileyDataObject) dataObject).toStorageFormat());
        }
    }

    /**
     * Remove a value for the Metadata field which is linked to this type from the given content.
     *
     * @param contentEntityObject the content from which the Metadata field value should be removed
     */
    @Override
    public void removeDataObject(ContentEntityObject contentEntityObject) {
        contentPropertyManager.removeProperty(contentEntityObject, getMetadataField().getKey());
    }

    @Override
    public UISupport getCqlUiSupport() {
        //NOTE: This is just a simple example how the cql integration could be implemented
        //Showing the real smiley faces in the UI would be possible,but it is a bit more complicated
        UISupport.Builder uiSupportBuilder = UISupport.builder().valueType(
                ValueType.STRING_TYPE).i18nKey(HtmlUtil.htmlEncode(getMetadataField().getTitle()) + (getMetadataField().getSpace() != null ?
                " (" + getMetadataField().getSpace().getKey() + ")" : ""));
        uiSupportBuilder.defaultOperator("=");

        uiSupportBuilder.dataUri(
                "/rest/communardo/metadata-integration-example/latest/smiley-values/" + getMetadataField().getId());
        return uiSupportBuilder.build();
    }

    @Deprecated
    @Override
    public void importTypeData(String exportedConfiguration) throws Exception {
        importTypeConfiguration(exportedConfiguration);
    }

    @Override
    public DataObject copyDataObject(DataObject sourceDataObject, MetadataField targetMetadataField) {
        // the data object is more or less a simple ID so we can just return the source object
        return sourceDataObject;
    }

    @Override
    public DataObject createDataObject(String exportFormat) throws MetadataImportException {
        return SmileyDataObject.fromExportFormat(exportFormat);
    }

    @Override
    public String exportDataObject(ContentEntityObject contentEntityObject) throws MetadataExportException {
        SmileyDataObject value = (SmileyDataObject) loadDataObject(contentEntityObject);
        if (value != null) {
            return value.toExportFormat();
        }
        return null;
    }

    @Override
    public String exportTypeConfiguration() throws MetadataExportException {
        return convertSmileyStyleToJson(loadSmileyStyle());
    }

    @Override
    public void importTypeConfiguration(String exportedConfiguration) throws MetadataImportException {
        SmileyStyle smileyTypeConfig = parseSmileyTypeConfigurationJson(exportedConfiguration);
        Map<String, String> typeConfiguration = loadTypeConfiguration();
        typeConfiguration.put(FIELD_CONFIG_SMILEY_STYLE, smileyTypeConfig.toString());
        updateTypeConfiguration(typeConfiguration);
    }

    public void setYellowTypeConfiguration() {
        Map<String, String> config = new HashMap<>();
        config.put(FIELD_CONFIG_SMILEY_STYLE, SmileyStyle.YELLOW.toString());
        updateTypeConfiguration(config);
    }

    private String convertSmileyStyleToJson(SmileyStyle smileyStyle) {
        JsonObject json = new JsonObject();
        json.addProperty(FIELD_CONFIG_SMILEY_STYLE, smileyStyle.toString());
        Gson gson = new Gson();
        return gson.toJson(json);
    }

    private SmileyStyle parseSmileyTypeConfigurationJson(String smileyTypeConfigJson) throws MetadataImportException {
        Gson gson = new Gson();
        try {
            JsonObject config = gson.fromJson(smileyTypeConfigJson, JsonObject.class);
            return resolveSmileyStyle(config.get(FIELD_CONFIG_SMILEY_STYLE).getAsString());
        } catch (JsonSyntaxException | UnsupportedOperationException e) {
            throw new MetadataImportException("Parsing the smiley type configuration JSON failed.", e);
        }
    }

    private String renderSmileyValueEditView(MetadataField metadataField, DataObject object) {
        Map<String, Object> defaultVelocityContext = MacroUtils.defaultVelocityContext();
        defaultVelocityContext.put("currentSmiley", object);
        defaultVelocityContext.put("allSmilies", SmileyDataObject.getAllSmilies());
        defaultVelocityContext.put("smileyStyle", loadSmileyStyle(metadataField).toString());
        defaultVelocityContext.put("metadataField", metadataField);
        defaultVelocityContext.put("i18n", i18nResolver);

        return VelocityUtils.getRenderedTemplate("template/smiley_type_edit.vm", defaultVelocityContext);
    }

    @Override
    public DataObject loadDefaultValue(MetadataSet metadataSet, MetadataField metadataField) {
        String defaultValue = (String) bandanaManager.getValue(new ConfluenceBandanaContext(), getSmileyDefaultValueBandanaKey(metadataSet, metadataField));
        return SmileyDataObject.fromStorageFormat(defaultValue);
    }

    private String getSmileyDefaultValueBandanaKey(MetadataSet metadataSet, MetadataField metadataField) {
        return "smiley-default-value" + metadataField.getId() + "_" + metadataSet.getId();
    }

    @Override
    public void saveDefaultValue(Map<String, String[]> parameters, MetadataSet metadataSet, MetadataField metadataField) {
        DataObject dataObject = createDataObject(parameters);
        saveDefaultValue(metadataSet, metadataField, dataObject);
    }

    @Override
    public void saveDefaultValue(MetadataSet metadataSet, MetadataField metadataField, DataObject dataObject) {
        if (dataObject instanceof SmileyDataObject) {
            String storedDefaultValue = ((SmileyDataObject) dataObject).toStorageFormat();
            bandanaManager.setValue(new ConfluenceBandanaContext(), getSmileyDefaultValueBandanaKey(metadataSet, metadataField), storedDefaultValue);
        }
    }

    @Override
    public void deleteDefaultValue(MetadataSet metadataSet, MetadataField metadataField) {
        bandanaManager.removeValue(new ConfluenceBandanaContext(), getSmileyDefaultValueBandanaKey(metadataSet, metadataField));
    }

    @Override
    public String renderDefaultValueEdit(MetadataSet metadataSet, MetadataField metadataField, DataObject object) {
        return renderSmileyValueEditView(metadataField, object);
    }

    @Override
    public String exportDefaultValueConfiguration(MetadataSet metadataSet, MetadataField metadataField) throws MetadataExportException {
        SmileyDataObject defaultValue = (SmileyDataObject) loadDefaultValue(metadataSet, metadataField);
        return defaultValue != null ? defaultValue.toExportFormat() : null;
    }

    @Override
    public void importDefaultValueConfiguration(MetadataSet metadataSet, MetadataField metadataField, String exportedConfiguration) throws MetadataImportException {
        if (StringUtils.isNotBlank(exportedConfiguration)) {
            saveDefaultValue(metadataSet, metadataField, SmileyDataObject.fromExportFormat(exportedConfiguration));
        }
    }
}
